<?php
require 'conexao.php';

// Habilita exibição de erros para debug
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Configuração de CORS
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// Verifica se é uma requisição OPTIONS (pré-voo do CORS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Define que a resposta será JSON
header("Content-Type: application/json");

// Converte entrada JSON
$json = file_get_contents('php://input');
$data = json_decode($json, true);

// Caminho para salvar logs de erro
$logFile = __DIR__ . '/logs/api_errors.log';

// Função para registrar erros
function logError($message) {
    global $logFile;
    file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] " . $message . PHP_EOL, FILE_APPEND);
}

try {
    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
        case 'GET':
            handleGet();
            break;
        case 'POST':
            handlePost($data);
            break;
        case 'PUT':
            handlePut($data);
            break;
        case 'DELETE':
            handleDelete($data);
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            break;
    }
} catch (Exception $e) {
    logError("Erro: " . $e->getMessage());
    http_response_code(400);
    echo json_encode(['error' => $e->getMessage()]);
}

// ===================== FUNÇÕES DE MANIPULAÇÃO DE REQUISIÇÕES ===================== //

function handleGet() {
    global $pdo;

    if (isset($_GET['id'])) {
        $id = intval($_GET['id']);
        $stmt = $pdo->prepare("SELECT * FROM boleto_duplicatas WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
    } elseif (isset($_GET['clientes'])) {
        fetchClientes($_GET['clientes']);
    } else {
        fetchBoletos();
    }
}

// Função para buscar clientes na tabela clientes_novo
function fetchClientes($query) {
    global $pdo;
    
    try {
        $busca = '%' . htmlspecialchars($query) . '%';
        $stmt = $pdo->prepare("SELECT id, nome_completo, cpf_cnpj FROM clientes_novo WHERE nome_completo LIKE :busca OR cpf_cnpj LIKE :busca LIMIT 10");
        $stmt->bindParam(':busca', $busca, PDO::PARAM_STR);
        $stmt->execute();
        echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
    } catch (Exception $e) {
        logError("Erro ao buscar clientes: " . $e->getMessage());
        echo json_encode(['error' => 'Erro ao buscar clientes.']);
    }
}

// Função para buscar boletos
function fetchBoletos() {
    global $pdo;

    try {
        $stmt = $pdo->prepare("SELECT id, documento, 
            DATE_FORMAT(emissao, '%Y-%m-%d') as emissao, 
            DATE_FORMAT(vencimento, '%Y-%m-%d') as vencimento, 
            cliente, 
            cpf_cnpj, 
            valor,  -- Removido FORMAT aqui
            empresa, 
            juros,  -- Removido FORMAT aqui
            multa,  -- Removido FORMAT aqui
            protesto 
            FROM boleto_duplicatas 
            ORDER BY id DESC");

        $stmt->execute();
        
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($result);
    } catch (Exception $e) {
        logError("Erro ao buscar boletos: " . $e->getMessage());
        echo json_encode(['error' => 'Erro ao buscar boletos.']);
    }
}


function handlePost($data) {
    global $pdo;

    try {
        $stmt = $pdo->prepare("INSERT INTO boleto_duplicatas 
            (documento, emissao, vencimento, cliente, cpf_cnpj, valor, empresa, juros, multa, protesto, empresa_id, usuario_id)
            VALUES (:documento, :emissao, :vencimento, :cliente, :cpf_cnpj, :valor, :empresa, :juros, :multa, :protesto, :empresa_id, :usuario_id)");

        $stmt->execute([
            ":documento" => $data['documento'] ?? null,
            ":emissao" => $data['emissao'] ?? null,
            ":vencimento" => $data['vencimento'] ?? null,
            ":cliente" => $data['cliente'] ?? null,
            ":cpf_cnpj" => !empty($data['cpf_cnpj']) ? $data['cpf_cnpj'] : null, // Permite CPF/CNPJ nulo
            ":valor" => isset($data['valor']) ? floatval(str_replace(',', '.', $data['valor'])) : null, // Converte vírgula para ponto
            ":empresa" => $data['empresa'] ?? null,
            ":juros" => isset($data['juros']) ? floatval(str_replace(',', '.', $data['juros'])) : 0.00,
            ":multa" => isset($data['multa']) ? floatval(str_replace(',', '.', $data['multa'])) : 0.00,
            ":protesto" => $data['protesto'] ?? null,
            ":empresa_id" => $data['empresa_id'] ?? null,
            ":usuario_id" => $data['usuario_id'] ?? null,
        ]);

        echo json_encode(['message' => 'Boleto cadastrado com sucesso.', 'id' => $pdo->lastInsertId()]);
    } catch (Exception $e) {
        logError("Erro ao cadastrar boleto: " . $e->getMessage());
        echo json_encode(['error' => 'Erro ao cadastrar boleto.']);
    }
}

function handlePut($data) {
    global $pdo;

    try {
        if (!$data || !isset($data['id'])) {
            throw new Exception("ID do boleto não foi informado para atualização.");
        }

        $stmt = $pdo->prepare("UPDATE boleto_duplicatas SET documento = :documento, emissao = :emissao, vencimento = :vencimento, cliente = :cliente, 
                               cpf_cnpj = :cpf_cnpj, valor = :valor, empresa = :empresa, juros = :juros, multa = :multa, protesto = :protesto 
                               WHERE id = :id");

        $result = $stmt->execute([
            ":documento" => $data['documento'] ?? null,
            ":emissao" => $data['emissao'] ?? null,
            ":vencimento" => $data['vencimento'] ?? null,
            ":cliente" => $data['cliente'] ?? null,
            ":cpf_cnpj" => $data['cpf_cnpj'] ?? null,
            ":valor" => isset($data['valor']) ? floatval($data['valor']) : null,
            ":empresa" => $data['empresa'] ?? null,
            ":juros" => isset($data['juros']) ? floatval($data['juros']) : 0.00,
            ":multa" => isset($data['multa']) ? floatval($data['multa']) : 0.00,
            ":protesto" => $data['protesto'] ?? null,
            ":id" => intval($data['id']),
        ]);

        if ($result) {
            echo json_encode(['message' => 'Boleto atualizado com sucesso.']);
        } else {
            throw new Exception("Erro ao atualizar boleto.");
        }
    } catch (Exception $e) {
        logError("Erro ao atualizar boleto: " . $e->getMessage());
        echo json_encode(['error' => 'Erro ao atualizar boleto.']);
    }
}

function handleDelete($data) {
    global $pdo;

    try {
        if (!$data || !isset($data['id'])) {
            throw new Exception("ID do boleto não foi informado para exclusão.");
        }

        $stmt = $pdo->prepare("DELETE FROM boleto_duplicatas WHERE id = :id");
        $stmt->bindParam(':id', $data['id'], PDO::PARAM_INT);
        $stmt->execute();

        echo json_encode(['message' => 'Boleto excluído com sucesso.']);
    } catch (Exception $e) {
        logError("Erro ao excluir boleto: " . $e->getMessage());
        echo json_encode(['error' => 'Erro ao excluir boleto.']);
    }
}
?>