<?php
include 'conexao.php';

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json');

// Tratamento para requisições OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Função para log de mensagens
function logMessage($message) {
    $logFile = 'log.txt';
    $date = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$date] $message" . PHP_EOL, FILE_APPEND);
}

$method = $_SERVER['REQUEST_METHOD'];

// Verifica conexão com o banco
if (!isset($pdo)) {
    http_response_code(500);
    echo json_encode(['message' => 'Erro na conexão com o banco de dados.']);
    logMessage("Erro: Conexão com o banco de dados falhou.");
    exit;
}

// Função para validar campos obrigatórios
function validateFields($data, $fields) {
    foreach ($fields as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            return ['valid' => false, 'field' => $field];
        }
    }
    return ['valid' => true];
}

// Roteamento baseado no método HTTP
switch ($method) {
    case 'GET':
        if (isset($_GET['id'])) {
            // Obter uma conta específica
            $id = intval($_GET['id']);
            try {
                $sql = "SELECT 
                            cp.id, 
                            cp.fornecedor_id, 
                            f.nome_razao_social AS fornecedor_nome,
                            cp.categoria_id, 
                            ct.nome AS categoria_nome,
                            cp.valor, 
                            cp.data_vencimento, 
                            cp.status, 
                            cp.historico,
                            cp.data_pagamento,
                            cp.data_emissao,
                            cp.numero_nota_fiscal,
                            cp.numero_pedido,
                            cp.parcelado,
                            cp.quantidade_parcelas
                        FROM contas_pagar AS cp
                        LEFT JOIN fornecedores_novo AS f ON cp.fornecedor_id = f.id
                        LEFT JOIN categorias_transacoes AS ct ON cp.categoria_id = ct.id
                        WHERE cp.id = :id";
                $stmt = $pdo->prepare($sql);
                $stmt->bindValue(':id', $id, PDO::PARAM_INT);
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($result) {
                    echo json_encode($result);
                    logMessage("GET: Conta ID $id obtida com sucesso.");
                } else {
                    http_response_code(404);
                    echo json_encode(['message' => 'Conta não encontrada.']);
                    logMessage("GET: Conta ID $id não encontrada.");
                }
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(['message' => 'Erro ao obter conta.', 'error' => $e->getMessage()]);
                logMessage("Erro ao obter conta ID $id: " . $e->getMessage());
            }
        } else {
            // Obter todas as contas
            try {
                $sql = "SELECT 
                            cp.id, 
                            cp.fornecedor_id, 
                            f.nome_razao_social AS fornecedor_nome,
                            cp.categoria_id, 
                            ct.nome AS categoria_nome,
                            cp.valor, 
                            cp.data_vencimento, 
                            cp.status, 
                            cp.historico,
                            cp.data_pagamento,
                            cp.data_emissao,
                            cp.numero_nota_fiscal,
                            cp.numero_pedido,
                            cp.parcelado,
                            cp.quantidade_parcelas
                        FROM contas_pagar AS cp
                        LEFT JOIN fornecedores_novo AS f ON cp.fornecedor_id = f.id
                        LEFT JOIN categorias_transacoes AS ct ON cp.categoria_id = ct.id";
                $stmt = $pdo->prepare($sql);
                $stmt->execute();
                $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
                echo json_encode($result);
                logMessage("GET: Contas a pagar listadas com sucesso.");
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(['message' => 'Erro ao listar contas a pagar.', 'error' => $e->getMessage()]);
                logMessage("Erro ao listar contas a pagar: " . $e->getMessage());
            }
        }
        break;

    case 'POST':
        $data = json_decode(file_get_contents("php://input"), true);
        logMessage("Recebido POST: " . json_encode($data));

        $validation = validateFields($data, ['fornecedor_id', 'categoria_id', 'valor']);
        if (!$validation['valid']) {
            http_response_code(400);
            echo json_encode(['message' => 'Campo obrigatório ausente: ' . $validation['field']]);
            logMessage("Erro: Campo obrigatório ausente (" . $validation['field'] . ")");
            exit;
        }

        $sql = "INSERT INTO contas_pagar 
                (fornecedor_id, categoria_id, valor, data_vencimento, status, historico, data_pagamento, data_emissao, numero_nota_fiscal, numero_pedido, parcelado, quantidade_parcelas) 
                VALUES 
                (:fornecedor_id, :categoria_id, :valor, :data_vencimento, :status, :historico, :data_pagamento, :data_emissao, :numero_nota_fiscal, :numero_pedido, :parcelado, :quantidade_parcelas)";
        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':fornecedor_id' => $data['fornecedor_id'],
                ':categoria_id' => $data['categoria_id'],
                ':valor' => $data['valor'],
                ':data_vencimento' => $data['data_vencimento'] ?? null,
                ':status' => $data['status'] ?? 'pendente',
                ':historico' => $data['historico'] ?? null,
                ':data_pagamento' => $data['data_pagamento'] ?? null,
                ':data_emissao' => $data['data_emissao'] ?? null,
                ':numero_nota_fiscal' => $data['numero_nota_fiscal'] ?? null,
                ':numero_pedido' => $data['numero_pedido'] ?? null,
                ':parcelado' => $data['parcelado'] ?? 'nao',
                ':quantidade_parcelas' => $data['quantidade_parcelas'] ?? null
            ]);
            echo json_encode(['message' => 'Conta a pagar adicionada com sucesso.']);
            logMessage("POST: Conta a pagar adicionada com sucesso.");
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['message' => 'Erro ao adicionar conta a pagar.', 'error' => $e->getMessage()]);
            logMessage("Erro ao adicionar conta a pagar: " . $e->getMessage());
        }
        break;
        
        case 'GET':
    if (isset($_GET['resumo'])) {
        try {
            $sqlReceitas = "SELECT SUM(valor) AS total FROM contas_pagar WHERE status = 'pago'";
            $stmtReceitas = $pdo->prepare($sqlReceitas);
            $stmtReceitas->execute();
            $receitas = $stmtReceitas->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

            $sqlDespesas = "SELECT SUM(valor) AS total FROM contas_pagar WHERE status != 'pago'";
            $stmtDespesas = $pdo->prepare($sqlDespesas);
            $stmtDespesas->execute();
            $despesas = $stmtDespesas->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

            echo json_encode([
                'receitas' => $receitas,
                'despesas' => $despesas,
                'total' => $receitas - $despesas
            ]);
            logMessage("GET: Resumo gerado com sucesso.");
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['message' => 'Erro ao gerar resumo.', 'error' => $e->getMessage()]);
            logMessage("Erro ao gerar resumo: " . $e->getMessage());
        }
        exit;
    }


    case 'PUT':
        $data = json_decode(file_get_contents("php://input"), true);
        $id = $data['id'] ?? null;
        logMessage("Recebido PUT: " . json_encode($data));

        if (!$id) {
            http_response_code(400);
            echo json_encode(['message' => 'ID é obrigatório para atualizar a conta.']);
            logMessage("Erro: ID ausente no PUT.");
            exit;
        }

        $sql = "UPDATE contas_pagar 
                SET fornecedor_id = :fornecedor_id, categoria_id = :categoria_id, valor = :valor, data_vencimento = :data_vencimento, 
                    status = :status, historico = :historico, data_pagamento = :data_pagamento, data_emissao = :data_emissao, 
                    numero_nota_fiscal = :numero_nota_fiscal, numero_pedido = :numero_pedido, parcelado = :parcelado, quantidade_parcelas = :quantidade_parcelas 
                WHERE id = :id";
        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':fornecedor_id' => $data['fornecedor_id'],
                ':categoria_id' => $data['categoria_id'],
                ':valor' => $data['valor'],
                ':data_vencimento' => $data['data_vencimento'] ?? null,
                ':status' => $data['status'] ?? 'pendente',
                ':historico' => $data['historico'] ?? null,
                ':data_pagamento' => $data['data_pagamento'] ?? null,
                ':data_emissao' => $data['data_emissao'] ?? null,
                ':numero_nota_fiscal' => $data['numero_nota_fiscal'] ?? null,
                ':numero_pedido' => $data['numero_pedido'] ?? null,
                ':parcelado' => $data['parcelado'] ?? 'nao',
                ':quantidade_parcelas' => $data['quantidade_parcelas'] ?? null,
                ':id' => $id
            ]);
            echo json_encode(['message' => 'Conta a pagar atualizada com sucesso.']);
            logMessage("PUT: Conta a pagar ID $id atualizada com sucesso.");
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['message' => 'Erro ao atualizar conta a pagar.', 'error' => $e->getMessage()]);
            logMessage("Erro ao atualizar conta a pagar ID $id: " . $e->getMessage());
        }
        break;

    case 'DELETE':
        $data = json_decode(file_get_contents("php://input"), true);
        $id = $data['id'] ?? null;
        logMessage("Recebido DELETE: ID = " . json_encode($id));

        if (!$id) {
            http_response_code(400);
            echo json_encode(['message' => 'ID é obrigatório para excluir a conta.']);
            logMessage("Erro: ID ausente no DELETE.");
            exit;
        }

        $sql = "DELETE FROM contas_pagar WHERE id = :id";
        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':id' => $id]);
            echo json_encode(['message' => 'Conta a pagar excluída com sucesso.']);
            logMessage("DELETE: Conta a pagar ID $id excluída com sucesso.");
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['message' => 'Erro ao excluir conta a pagar.', 'error' => $e->getMessage()]);
            logMessage("Erro ao excluir conta a pagar ID $id: " . $e->getMessage());
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['message' => 'Método não permitido.']);
        logMessage("Erro: Método HTTP $method não permitido.");
        break;
}
?>
