<?php
include 'conexao.php';

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json');

// Tratamento para requisições OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Função para log de mensagens
function logMessage($message) {
    $logFile = 'log.txt';
    $date = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$date] $message" . PHP_EOL, FILE_APPEND);
}

$method = $_SERVER['REQUEST_METHOD'];

// Verifica conexão com o banco
if (!isset($pdo)) {
    http_response_code(500);
    echo json_encode(['message' => 'Erro na conexão com o banco de dados.']);
    logMessage("Erro: Conexão com o banco de dados falhou.");
    exit();
}

// Endpoint para listar clientes
if ($method === 'GET' && isset($_GET['clientes'])) {
    try {
        $sql = "SELECT id, nome_razao_social FROM clientes";
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($result);
        logMessage("GET: Lista de clientes obtida com sucesso.");
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['message' => 'Erro ao listar clientes.', 'error' => $e->getMessage()]);
        logMessage("Erro ao listar clientes: " . $e->getMessage());
    }
    exit();
}

// Roteamento baseado no método HTTP
switch ($method) {
    case 'GET':
        if (isset($_GET['id'])) {
            // Obter uma conta específica
            $id = intval($_GET['id']);
            try {
                $sql = "SELECT 
                            cr.id, 
                            cr.cliente_id, 
                            cl.nome_razao_social AS cliente_nome, 
                            cr.categoria_id, 
                            ct.nome AS categoria_nome, 
                            cr.valor, 
                            cr.data_vencimento, 
                            cr.status, 
                            cr.data_recebimento
                        FROM contas_receber AS cr
                        LEFT JOIN clientes AS cl ON cr.cliente_id = cl.id
                        LEFT JOIN categorias_transacoes AS ct ON cr.categoria_id = ct.id
                        WHERE cr.id = :id";
                $stmt = $pdo->prepare($sql);
                $stmt->bindValue(':id', $id, PDO::PARAM_INT);
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($result) {
                    echo json_encode($result);
                    logMessage("GET: Conta a receber ID $id obtida com sucesso.");
                } else {
                    http_response_code(404);
                    echo json_encode(['message' => 'Conta não encontrada.']);
                    logMessage("GET: Conta a receber ID $id não encontrada.");
                }
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(['message' => 'Erro ao obter conta.', 'error' => $e->getMessage()]);
                logMessage("Erro ao obter conta ID $id: " . $e->getMessage());
            }
        } else {
            // Obter todas as contas
            try {
                $sql = "SELECT 
                            cr.id, 
                            cr.cliente_id, 
                            cl.nome_razao_social AS cliente_nome, 
                            cr.categoria_id, 
                            ct.nome AS categoria_nome, 
                            cr.valor, 
                            cr.data_vencimento, 
                            cr.status, 
                            cr.data_recebimento
                        FROM contas_receber AS cr
                        LEFT JOIN clientes AS cl ON cr.cliente_id = cl.id
                        LEFT JOIN categorias_transacoes AS ct ON cr.categoria_id = ct.id";
                $stmt = $pdo->prepare($sql);
                $stmt->execute();
                $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
                echo json_encode($result);
                logMessage("GET: Contas a receber listadas com sucesso.");
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(['message' => 'Erro ao listar contas a receber.', 'error' => $e->getMessage()]);
                logMessage("Erro ao listar contas a receber: " . $e->getMessage());
            }
        }
        break;

    case 'POST':
        $data = json_decode(file_get_contents("php://input"), true);

        // Validação de campos obrigatórios
        if (empty($data['cliente_id']) || empty($data['categoria_id']) || empty($data['valor'])) {
            http_response_code(400);
            echo json_encode(['message' => 'Campos obrigatórios ausentes.']);
            logMessage("POST: Erro - Campos obrigatórios ausentes.");
            exit();
        }

        $sql = "INSERT INTO contas_receber 
                (cliente_id, categoria_id, valor, data_vencimento, status, data_recebimento) 
                VALUES (:cliente_id, :categoria_id, :valor, :data_vencimento, :status, :data_recebimento)";
        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':cliente_id' => $data['cliente_id'],
                ':categoria_id' => $data['categoria_id'],
                ':valor' => $data['valor'],
                ':data_vencimento' => $data['data_vencimento'] ?? null,
                ':status' => $data['status'] ?? 'pendente',
                ':data_recebimento' => $data['data_recebimento'] ?? null,
            ]);
            echo json_encode(['message' => 'Conta a receber adicionada com sucesso.']);
            logMessage("POST: Conta a receber adicionada com sucesso.");
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['message' => 'Erro ao adicionar conta a receber.', 'error' => $e->getMessage()]);
            logMessage("Erro ao adicionar conta a receber: " . $e->getMessage());
        }
        break;

    case 'PUT':
        $data = json_decode(file_get_contents("php://input"), true);
        $id = $data['id'] ?? null;
        if (!$id) {
            http_response_code(400);
            echo json_encode(['message' => 'ID é obrigatório para atualizar a conta.']);
            exit();
        }

        $sql = "UPDATE contas_receber 
                SET cliente_id = :cliente_id, categoria_id = :categoria_id, valor = :valor, 
                    data_vencimento = :data_vencimento, status = :status, data_recebimento = :data_recebimento
                WHERE id = :id";
        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':cliente_id' => $data['cliente_id'],
                ':categoria_id' => $data['categoria_id'],
                ':valor' => $data['valor'],
                ':data_vencimento' => $data['data_vencimento'] ?? null,
                ':status' => $data['status'] ?? 'pendente',
                ':data_recebimento' => $data['data_recebimento'] ?? null,
                ':id' => $id,
            ]);
            echo json_encode(['message' => 'Conta a receber atualizada com sucesso.']);
            logMessage("PUT: Conta a receber ID $id atualizada com sucesso.");
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['message' => 'Erro ao atualizar conta a receber.', 'error' => $e->getMessage()]);
            logMessage("Erro ao atualizar conta ID $id: " . $e->getMessage());
        }
        break;

    case 'DELETE':
        $data = json_decode(file_get_contents("php://input"), true);
        $id = $data['id'] ?? null;
        if (!$id) {
            http_response_code(400);
            echo json_encode(['message' => 'ID é obrigatório para excluir a conta.']);
            exit();
        }

        $sql = "DELETE FROM contas_receber WHERE id = :id";
        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':id' => $id]);
            echo json_encode(['message' => 'Conta a receber excluída com sucesso.']);
            logMessage("DELETE: Conta a receber ID $id excluída com sucesso.");
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['message' => 'Erro ao excluir conta a receber.', 'error' => $e->getMessage()]);
            logMessage("Erro ao excluir conta ID $id: " . $e->getMessage());
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['message' => 'Método não permitido.']);
        logMessage("Erro: Método HTTP $method não permitido.");
        break;
}
?>
